#ifndef __CTextEdit__
#define __CTextEdit__

/** @cond */
#ifdef WIN32
	#pragma warning(disable : 4311)	// pointer truncation from 'Exponent::GUI::Controls::CTextEdit *const ' to 'long'
	#pragma warning(disable : 4312)	// conversion from 'LONG' to 'Exponent::GUI::Controls::CTextEdit *' of greater size
#endif
/** @endcond */

#include "CTextLabel.hpp"
#include "../Basics/CColour.hpp"
#include "../Listeners/IActionListener.hpp"

//	===========================================================================

using Exponent::GUI::Basics::CColour;
using Exponent::GUI::Controls::CTextLabel;
using Exponent::GUI::Listeners::IActionListener;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CTextEdit CTextEdit.hpp
			 * @brief Displays text in a box area and allows the user to edit it
			 *
			 * @date 13/04/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CTextEdit.hpp,v 1.7 2007/02/08 21:07:54 paul Exp $
			 */
			class CTextEdit : public CTextLabel, public IActionListener
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================

			public:

//	===========================================================================

				/**
				 * @enum ETextEditAlignment
				 * @brief Alignment of the text edit
				 */
				enum ETextEditAlignment 
				{
					/** @cond */
					#ifdef WIN32
						e_center = ES_CENTER,
						e_left   = ES_LEFT,
						e_right  = ES_RIGHT,
					#else
					/** @endcond */
						e_center = 0,		/**< Center align text */
						e_left,				/**< Left align text */
						e_right				/**< Right align text */
					/** @cond */
					#endif
					/** @endcond */
				};

//	===========================================================================

				/**
				 * Construction
				 * @param root The root control that this control is being added to
				 * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required
				 * @param area The area of this control relative to the root controls top left
				 * @param text The text to display
				 * @param listener The action listener
				 */
				CTextEdit(IControlRoot *root, const long uniqueId, const CRect &area, const CString &text, IActionListener *listener = NULL);

				/**
				 * Destruction
				 */
				virtual ~CTextEdit();

//	===========================================================================

				/**
				 * Handle left button being clicked
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonDown(CMouseEvent &event);

				/**
				 * Handle a double click on the left button
				 * @param event The event to handle
				 */
				virtual void handleDoubleClick(CMouseEvent &event);

				/**
				 * Handle events - NOT INTENDED FOR PUBLIC USE!!
				 * @param event The event to handle
				 */
				virtual void handleActionEvent(const CActionEvent &event);

				/**
				 * Should we limit the text
				 * @param limitText If true text is limited to to textLimit numberof characters. If false then text is not limited
				 * @param textLimit If limitText is true, this parameter defines the number of limited characters
				 */
				virtual void limitText(const bool limitText = false, const long textLimit = 256);

				/**
				 * Set text alignment
				 * @param alignment The alignment to draw the text with
				 */
				virtual void setAlignment(const ETextEditAlignment alignment) { m_alignment = alignment; }

				/**
				 * Are we editable?
				 * @param editable If true the text is editable, if false this becomesa CTextLabel
				 * @see CTextLabel
				 */
				virtual void setEditable(const bool editable = true) { m_isEditable = editable; }

				/**
				 * Set single click editable
				 * @param editable If true then a single click will trigger an edit event. A double click will not be handlded in this case
				 */
				virtual void setSingleClickEditable(const bool editable = true) { m_singleClickEdit = editable; }

				/**
				 * Get the string
				 * @retval const CString & The text
				 */
				virtual const CString &getString() const { return m_text; }
				
				/** @cond */
				#ifdef WIN32
				/** @endcond */
					
					/**
					 * Dirty little hack in to the edit window, so that we can correctly set the background colour fo the box.
					 * @param drawContext The drawing context for the edit box
					 * @retval LRESULT Handle to the brush
					 */
					LRESULT setEditWindowDrawConditions(HDC drawContext);
				
				/** @cond */
				#else
				/** @endcond */

					/**
					 * Get the TXNObject associated with the edit
					 * @retval TXNObject The editor handle
					 */
					virtual TXNObject getTxnHandle() const { return m_txnHandle; }
					
					/**
					 * Is this a cancel call to handleActionEvent?
					 * @param cancel True if is a cancel call
					 * @note Internal use only, do not call this function
					 */
					virtual void cancelCall(const bool cancel) { m_cancel = cancel; }

					/**
					 * Draw the controls
					 * @param graphics The graphics context
					 */
					virtual void drawControl(CGraphics &graphics);
				/** @cond */
				#endif
				/** @endcond */

//	===========================================================================

			protected:

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					/**
					 * Call back function for the window
					 * @param windowHandle Handle to the edit window
					 * @param message The event to handle
					 * @param wParam The event high bits
					 * @param lParam The event low bits
					 * @retval LONG The window handler result
					 * @see WindowProc
					 */
					static LONG WINAPI windowProcTextEdit(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam);
				/** @cond */
				#else
				/** @endcond */
					/**
					 * Call back function for the window
					 * @param inHandlerCallRef Event handler reference
					 * @param inEvent The event to handle
					 * @param inUserData A reference to the CTextEdit that created the window
					 * @retval pascal OSStatus The status of the event
					 * @see TXNObject
					 */
					static pascal OSStatus windowProcTextEdit(EventHandlerCallRef inHandlerCallRef, EventRef inEvent, void *inUserData);
				/** @cond */
				#endif
				/** @endcond */

				/**
				 * Launch the edit
				 * @param event The mouse event that triggered
			     */
				void launchEdit(CMouseEvent &event);

//	===========================================================================

				/** @cond */
				#ifdef WIN32
				/** @endcond */
					HWND m_editWindowHandle;				/**< handle to the edit window */
					static WNDPROC m_lastWinProc;			/**< The main window proc */
					IControl *m_previouslyLockedControl;	/**< The last locked control */
					CBrush m_brush;							/**< The brush */
				/** @cond */
				#else
				/** @endcond */
					TXNObject m_txnHandle;					/**< Handle to the edit window */
					EventHandlerRef m_textEditEventHandler;	/**< Event handler */
					bool m_cancel;							/**< Is this  cancel call? */
				/** @cond */
				#endif
				/** @endcond */

				bool m_limitText;							/**< Should the text be limited */
				long m_textLimit;							/**< If so what is the character limit */
				ETextEditAlignment m_alignment;				/**< The alignment */
				bool m_isEditable;							/**< Can we be edited? */
				bool m_singleClickEdit;						/**< Can we be edited by a single click */
			};
		}
	}
}
#endif	// End of CTextEdit.hpp